// Copyright (c) 2017 Pieter Wuille
// Distributed under the MIT software license, see the accompanying
// file COPYING or http://www.opensource.org/licenses/mit-license.php.

#include <bech32.h>
#include <test/util/setup_common.h>
#include <test/util/str.h>

#include <boost/test/unit_test.hpp>

BOOST_FIXTURE_TEST_SUITE(bech32_tests, BasicTestingSetup)

BOOST_AUTO_TEST_CASE(bip173_testvectors_valid)
{
    static const std::string CASES[] = {
        "A12UEL5L",
        "a12uel5l",
        "an83characterlonghumanreadablepartthatcontainsthenumber1andtheexcludedcharactersbio1tt5tgs",
        "abcdef1qpzry9x8gf2tvdw0s3jn54khce6mua7lmqqqxw",
        "11qqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqc8247j",
        "split1checkupstagehandshakeupstreamerranterredcaperred2y9e3w",
        "?1ezyfcl",
    };
    for (const std::string& str : CASES) {
        auto ret = bech32::Decode(str);
        BOOST_CHECK(!ret.first.empty());
        std::string recode = bech32::Encode(ret.first, ret.second);
        BOOST_CHECK(!recode.empty());
        BOOST_CHECK(CaseInsensitiveEqual(str, recode));
    }
}

BOOST_AUTO_TEST_CASE(bip173_testvectors_invalid)
{
    static const std::string CASES[] = {
        " 1nwldj5",
        "\x7f""1axkwrx",
        "\x80""1eym55h",
        "an84characterslonghumanreadablepartthatcontainsthenumber1andtheexcludedcharactersbio1569pvx",
        "pzry9x0s0muk",
        "1pzry9x0s0muk",
        "x1b4n0q5v",
        "li1dgmt3",
        "de1lg7wt\xff",
        "A1G7SGD8",
        "10a06t8",
        "1qzzfhee",
        "a12UEL5L",
        "A12uEL5L",
    };
    static const std::pair<std::string, int> ERRORS[] = {
        {"Invalid character", 0},
        {"Invalid character", 0},
        {"Invalid character", 0},
        {"String too long", 90},
        {"Missing separator", 0},
        {"Invalid separator position", 0},
        {"Invalid Base 32 character", 2},
        {"Invalid separator position", 2},
        {"Invalid character", 8},
        {"Invalid", 0}, // The checksum is calculated using the uppercase form so the entire string is invalid, not just a few characters
        {"Invalid separator position", 0},
        {"Invalid separator position", 0},
        {"Invalid character", 3},
        {"Invalid character", 3},
    };
    int i = 0;
    for (const std::string& str : CASES) {
        const std::pair<std::string, int>& err = ERRORS[i];
        auto ret = bech32::Decode(str);
        BOOST_CHECK(ret.first.empty());
        std::string error;
        int pos = bech32::LocateError(str, error);
        BOOST_CHECK_EQUAL(err.first, error);
        BOOST_CHECK_EQUAL(err.second, pos);
        i++;
    }
}

BOOST_AUTO_TEST_SUITE_END()
